﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using VIRP.CRS.BLL;
using VIRP.CRS.BO;
using VIRP.EFR.BLL;
using VIRP.EFR.BO;
using VIRP.EFR.UTIL;
using VIRP_WEB.Custom.EFR;

namespace VIRP_WEB.Controls.Custom.EFR
{
    public partial class LabResultInterpretation : UserControl
    {
        private const string SigningCommentsProviderLetter = "A copy of the enclosed letter addressed to the patient has been mailed directly to the patient. If you have any questions, please do not hesitate to call me at (410) 605-7373.";
        private const string SigningCommentsPatientLetter = "We are also sending your results to your VA healthcare provider. Please contact your VA healthcare provider if you have questions and to discuss the need for follow-up.";
        private const string LetterFromDefault = "Medical Director, Toxic Embedded Fragment Surveillance Center (512/11DU) Baltimore VA Medical Center";

        private bool showBioMonFollowUp;

        public event EventHandler OnCancelClick;
        public event EventHandler OnSaveClick;
        public event EventHandler OnInterpretComplete;
        public event EventHandler OnPrintProviderLetter;
        public event EventHandler OnPrintPatientLetter;
        public event EventHandler OnAddPrintQueue;

        private Address Address = null;
        private Address pcpInfo;
        private Address vamcInfo;

        private INTERPRETATION_LETTER IntLetter = null;
        private Address UseItToCompareAddress = null;

        private readonly LabResultsInterpretationManager labResultsInterpretationManager;

        public LabResultInterpretation()
        {
            labResultsInterpretationManager = new LabResultsInterpretationManager();
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            if (!IsPostBack)
            {
                BioFollowupDueDateValidator.ValueToCompare = DateTime.Today.ToShortDateString();
                ReminderFutureDateValidator.ValueToCompare = DateTime.Today.ToShortDateString();
            }

            // Follow Up Biomonitoring Due Date and Reminder Date are not required if 
            // Need Biomonitoring Follow Up is set to no (rbNeedBiomonitoringFollowUp)
            if (rbNeedBiomonitoringFollowUp.SelectedValue == "0")
            {
                RequiredValidatorDueDate.Enabled = false;
                RequiredFieldValidatorReminderDate.Enabled = false;
            }
            else
            {
                RequiredValidatorDueDate.Enabled = true;
                RequiredFieldValidatorReminderDate.Enabled = true;
            }

            LoadObjectDataSourceProperties();
        }

        private void LoadObjectDataSourceProperties()
        {
            ObjectDataSourceSigningClinician.SelectParameters["CURRENT_USER"].DefaultValue = HttpContext.Current.User.Identity.Name;

            Control baseControl = Parent;
            EfrBasePage basePage = baseControl.Page as EfrBasePage;
            if (basePage != null)
                ObjectDataSourceSigningClinician.SelectParameters["CURRENT_REGISTRY_ID"].DefaultValue = basePage.CurrentRegistryId.ToString();
        }

        public bool ShowBioMonitoringFollowUpSection
        {
            set
            {
                PanelBioFollowUp.Visible = value;
                showBioMonFollowUp = value;
            }
        }

        public bool ShowFacilityBoxAndPCPVAMCPanels
        {
            set
            {
                ProvAddress.ShowFacilityBoxAndPCPVAMCPanels = value;
                PatientAddress.ShowFacilityBoxAndPCPVAMCPanels = false;
            }
        }

        public bool ShowPrimaryAltAddressPanels
        {
            set
            {
                ProvAddress.ShowPrimaryAlternateAddressPanels = value;
                PatientAddress.ShowFacilityBoxAndPCPVAMCPanels = false;
            }
        }

        public void UpdateAddresses(string CURRENT_USER, int CURRENT_REGISTRY_ID, VIRP.CRS.BO.REFERRAL referral)
        {
            UpdateProviderAddress(CURRENT_USER, CURRENT_REGISTRY_ID, referral);
        }

        public void FillProviderAddress(Address address)
        {
           ProvAddress.PopulateAddress(address);
        }

        public void FillPatientAddress(Address address)
        {
           PatientAddress.PopulateAddress(address);
        }

        public void FillPatientPCPInfo(Address pcpInfo)
        {
           ProvAddress.PopulatePCPInfo(pcpInfo);
        }

        public void FillPatientVAMCInfo(Address vamcInfo)
        {
           ProvAddress.PopulateVAMCInfo(vamcInfo);
        }

        public void FillPrimaryAddress(Address address)
        {
           PatientAddress.PopulatePrimaryAddressInfo(address);
        }

        public void FillSecondaryAddress(Address address)
        {
            PatientAddress.PopulateSecondaryAddressInfo(address);
        }

        public void PopulateBioFollowUp(string CURRENT_USER, int CURRENT_REGISTRY_ID, WKF_CASE wCase)
        {
            var activity = WKF_CASE_ACTIVITYManager.GetitemsByWorkflowCaseId(CURRENT_USER, CURRENT_REGISTRY_ID, wCase.WKF_CASE_ID)
                .Where(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation) && a.ARCHIVED_FLAG == false)
                .FirstOrDefault();

            if (activity != null)
            {
                if (activity.FOLLOWUP_REQUIRED_FLAG.HasValue)
                {
                    if (activity.STD_WKFACTIVITYSTS.CODE != "INTERPRETED")
                        rbNeedBiomonitoringFollowUp.SelectedIndex = rbNeedBiomonitoringFollowUp.Items.IndexOf(
                            rbNeedBiomonitoringFollowUp.Items.FindByValue(Convert.ToInt32(activity.FOLLOWUP_REQUIRED_FLAG.Value).ToString()));
                    else
                    {
                        if (activity.FOLLOWUP_PLAN_DATE.HasValue)
                            rbNeedBiomonitoringFollowUp.SelectedValue = "1";
                        else
                            rbNeedBiomonitoringFollowUp.SelectedValue = "0";
                    }
                }

                DateTime bioMonDueDate = DateTime.Today.AddYears(5);
                DateTime reminderDate = new DateTime(bioMonDueDate.AddMonths(-1).Year, bioMonDueDate.AddMonths(-1).Month, 1);

                if (activity.FOLLOWUP_PLAN_DATE.HasValue)
                    //if the activity has a value saved display it
                    txtBioMonDueDate.Text = activity.FOLLOWUP_PLAN_DATE.Value.ToString("d");
                else
                    txtBioMonDueDate.Text = bioMonDueDate.ToString("d");

                if (activity.FOLLOWUP_REMINDER_DATE.HasValue)
                    //if the activity has a value saved display it
                    txtReminderDate.Text = activity.FOLLOWUP_REMINDER_DATE.Value.ToString("d");
                else
                    txtReminderDate.Text = reminderDate.ToString("d");

                //follow-up is set 
                if (activity.FOLLOWUP_WKF_CASE_ID.HasValue)
                {
                    WKF_CASE followUp = VIRP.CRS.BLL.WKF_CASEManager.GetItem(CURRENT_USER, CURRENT_REGISTRY_ID, activity.FOLLOWUP_WKF_CASE_ID.Value);

                    LabelBioFollowUp.Text = "The Follow Up Biomonitoring was created on " + followUp.CREATED.ToString("d") + ", Workflow ID ";

                    LinkBioFollowUp.Visible = true;
                    LinkBioFollowUp.Text = followUp.WKF_CASE_ID.ToString();
                    LinkBioFollowUp.NavigateUrl = string.Format("~/Custom/EFR/Patients/PatientWorkflows.aspx?patientId={0}", followUp.PATIENT_ID);
                }
                else if (activity.FOLLOWUP_DECLINED_DATE.HasValue)
                {
                    LabelBioFollowUp.Text = "The Follow Up Biomonitoring has been declined on " + activity.FOLLOWUP_DECLINED_DATE.Value.ToString("d");

                    LinkBioFollowUp.Visible = false;
                }
                else
                {
                    LabelBioFollowUp.Visible = false;
                    LinkBioFollowUp.Visible = false;
                }
            }
        }

        private Address GetLetterProviderAddress(INTERPRETATION_LETTER letter)
        {
            if (letter != null && !string.IsNullOrEmpty(letter.PROVIDER_ADDR1))
                return AddressFactory.Create(letter, "PROVIDER");
            return null;
        }

        private Address GetPrimaryCareProviderAddress(REFERRAL_REG referral)
        {
            return AddressFactory.Create(referral, "PROVIDER");
        }

        public void PopulatePatientProviderDetails(string CURRENT_USER, int CURRENT_REGISTRY_ID, WKF_CASE wCase, string letterType)
        {
            var activity = WKF_CASE_ACTIVITYManager.GetitemsByWorkflowCaseId(CURRENT_USER, CURRENT_REGISTRY_ID, wCase.WKF_CASE_ID)
                .Where(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation) && a.ARCHIVED_FLAG == false)
                .FirstOrDefault();

            INTERPRETATION_LETTER letter = null;
            if (activity != null)
                letter = INTERPRETATION_LETTERManager.GetItemByWkfCaseActivityId(CURRENT_USER, CURRENT_REGISTRY_ID, activity.WKF_CASE_ACTIVITY_ID);

            HandleAddresses(CURRENT_USER, CURRENT_REGISTRY_ID, wCase.WKF_CASE_ID, activity != null ? activity.WKF_CASE_ACTIVITY_ID : (int?)null);

            SetControlVisibility(letterType);

            if (activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.New)
                btnPrintPatntLttr.Enabled = btnPrintPrvdrLttr.Enabled = btnAddPrintQueue.Enabled = false;
            else
                btnPrintPatntLttr.Enabled = btnPrintPrvdrLttr.Enabled = btnAddPrintQueue.Enabled = true;

            BindLetterItems(letter, letterType);

            ConfigureButtons(activity);
        }

        private void HandleAddresses(string CURRENT_USER, int CURRENT_REGISTRY_ID, int? WKF_CASE_ID, int? WKF_CASE_ACTIVITY_ID)
        {
            Dictionary<AddressType, Address> addressMap = labResultsInterpretationManager.GetAddressesForWorkflow(
                CURRENT_USER, CURRENT_REGISTRY_ID, WKF_CASE_ID, WKF_CASE_ACTIVITY_ID);

            var patientPrimary = addressMap[AddressType.PatientPrimary];
            var providerLetter = addressMap.ContainsKey(AddressType.ProviderLetter) ? addressMap[AddressType.ProviderLetter] : null;

            var patientSecondary = addressMap.ContainsKey(AddressType.PatientSecondary) 
                && addressMap[AddressType.PatientSecondary] != null
                ? addressMap[AddressType.PatientSecondary] : null;

            var patientLetter = addressMap.ContainsKey(AddressType.PatientLetter) 
                && addressMap[AddressType.PatientLetter] != null
                ? addressMap[AddressType.PatientLetter] : null;

            Address = patientLetter != null ? patientLetter
                : patientSecondary != null ? patientSecondary
                : patientPrimary;

            FillPatientAddress(Address);
            FillPrimaryAddress(patientPrimary);

            HandleSecondaryAddress(patientSecondary);
            HandlePcpAddress(addressMap);
            HandleVamcAddress(addressMap);

            // Populate provider updated address
            // Updated address available, populate it
            if (WKF_CASE_ACTIVITY_ID != null && providerLetter != null && providerLetter.AddressPart1 != null)
                Address = addressMap[AddressType.ProviderLetter];
            else // Pre-populate original address from REFERRAL
                Address = addressMap[AddressType.ProviderReferral];

            FillProviderAddress(Address);
        }

        private void HandleSecondaryAddress(Address patientSecondary)
        {
            if (patientSecondary != null)
                FillSecondaryAddress(patientSecondary);

            if (patientSecondary == null)
            {
                var primPanel = PatientAddress.FindControl("PrimaryAddressPanel") as Panel;

                var secPanel = PatientAddress.FindControl("AlternateAddressPanel") as Panel;

                if (primPanel != null && secPanel != null)
                {
                    secPanel.Visible = false;
                    primPanel.Visible = false;
                }
            }
        }

        private void HandlePcpAddress(Dictionary<AddressType, Address> addressMap)
        {
            pcpInfo = addressMap[AddressType.ProviderReferral];
            FillPatientPCPInfo(pcpInfo);
        }

        private void HandleVamcAddress(Dictionary<AddressType, Address> addressMap)
        {
            Address vamcPrimary = addressMap.ContainsKey(AddressType.ReferralVamcPrimary) ? addressMap[AddressType.ReferralVamcPrimary] : null;
            Address vamcMailing = addressMap.ContainsKey(AddressType.ReferralVamcMailing) ? addressMap[AddressType.ReferralVamcMailing] : null;
            Address vamStreet = addressMap.ContainsKey(AddressType.ReferralVamcStreet) ? addressMap[AddressType.ReferralVamcStreet] : null;

            if (vamcPrimary != null && !string.IsNullOrEmpty(vamcPrimary.AddressPart1))
                vamcInfo = vamcPrimary;
            else if (vamcMailing != null && !string.IsNullOrEmpty(vamcMailing.AddressPart1))
                vamcInfo = vamcMailing;
            else if (vamStreet != null && !string.IsNullOrEmpty(vamStreet.AddressPart1))
                vamcInfo = vamStreet;

            FillPatientVAMCInfo(vamcInfo);
        }

        private void SetControlVisibility(string letterType)
        {
            //Set visibility of controls based on workflow type
            switch (letterType)
            {
                case "Bio":
                    //FROM
                    TextBoxFrom.Visible = true;
                    LabelForTextBoxFrom.Visible = true;
                    //SUBJECT
                    TextBoxSubject.Visible = true;
                    LabelForTextBoxSubject.Visible = true;
                    //Signing Clinician
                    DropDownListSigningClinician.Visible = true;
                    LabelForDropDownListSigningClinician.Visible = true;
                    //CC
                    TextBoxCCName.Visible = true;
                    LabelForTextBoxCCName.Visible = true;
                    //Signing Comments Provider
                    TextBoxSigningCommentsProviderLetter.Visible = true;
                    LabelForTextBoxSigningCommentsProviderLetter.Visible = true;
                    //Signing Comments Patient
                    TextBoxSigningCommentsPatientLetter.Visible = true;
                    LabelforTextBoxSigningCommentsPatientLetter.Visible = true;
                    break;
                case "Fragment":
                    //FROM
                    TextBoxFrom.Visible = true;
                    LabelForTextBoxFrom.Visible = true;
                    //SUBJECT
                    TextBoxSubject.Visible = true;
                    LabelForTextBoxSubject.Visible = true;
                    //Signing Clinician
                    DropDownListSigningClinician.Visible = true;
                    LabelForDropDownListSigningClinician.Visible = true;
                    //CC
                    TextBoxCCName.Visible = true;
                    LabelForTextBoxCCName.Visible = true;
                    //Signing Comments Provider
                    TextBoxSigningCommentsProviderLetter.Visible = true;
                    LabelForTextBoxSigningCommentsProviderLetter.Visible = true;
                    //Signing Comments Patient
                    TextBoxSigningCommentsPatientLetter.Visible = true;
                    LabelforTextBoxSigningCommentsPatientLetter.Visible = true;
                    break;
            }
        }

        private void BindLetterItems(INTERPRETATION_LETTER letter, string letterType)
        {
            //Bind interpretation letter items
            if (letter != null)
            {
                // interpretation notes to patient.
                TextBoxInterpretationNotesToPatient.Text = letter.PATIENT_INTERPRETATION_TEXT;

                // interpretation notes to provider.
                TextBoxInterpretationNotesToProvider.Text = letter.PROVIDER_INTERPRETATION_TEXT;

                // FROM
                TextBoxFrom.Text = letter.FROM_ADDRESS;

                // SUBJECT
                TextBoxSubject.Text = letter.Letter_SUBJECT;

                // Signing Clinician
                if (letter.STD_SIGNING_CLINICIAN_ID.HasValue)
                    DropDownListSigningClinician.SelectedValue = letter.STD_SIGNING_CLINICIAN_ID.Value.ToString();

                // CC
                TextBoxCCName.Text = letter.CCAddress;

                // Signing Comments Provider
                TextBoxSigningCommentsProviderLetter.Text = letter.PROVIDER_COMMENTS;

                // Signing Comments Patient
                TextBoxSigningCommentsPatientLetter.Text = letter.PATIENT_COMMENT;
            }
            else
            {
                // interpretation letter does not exist and the page is opening a "new" letter so populate with default values

                // interpretation notes to patient
                TextBoxInterpretationNotesToPatient.Text = "";

                // interpretation notes to provider
                TextBoxInterpretationNotesToProvider.Text = "";

                // FROM
                TextBoxFrom.Text = LetterFromDefault;

                // Signing Clinician
                DropDownListSigningClinician.SelectedIndex = 1;

                // CC
                TextBoxCCName.Text = "";

                // Signing Comments Provider
                TextBoxSigningCommentsProviderLetter.Text = SigningCommentsProviderLetter;
                
                // Signing Comments Patient
                TextBoxSigningCommentsPatientLetter.Text = SigningCommentsPatientLetter;

                // SUBJECT
                if (letterType.Equals("Bio", StringComparison.OrdinalIgnoreCase))
                    TextBoxSubject.Text = "Interpretation Urine Biomonitoring Results for";
                else if (letterType.Equals("Fragment", StringComparison.OrdinalIgnoreCase))
                    TextBoxSubject.Text = "Interpretation Fragment Analysis Results for";
            }
        }

        private void ConfigureButtons(WKF_CASE_ACTIVITY activity)
        {
            // Configure buttons
            if (activity.STD_WKFACTIVITYSTS.CODE == null)
                return;

            switch (activity.STD_WKFACTIVITYSTS.CODE)
            {
                case "NEW":

                    btnPrintPatntLttr.Text = " Preview Patient Letter ";
                    btnPrintPrvdrLttr.Text = "Preview Provider Letter";
                    btnSave.Enabled = true;
                    btnInterpretComplete.Enabled = true;
                    HiddenFieldCompletedInterpretation.Value = "0";
                    break;
                case "IN-PROCESS":
                    btnPrintPatntLttr.Text = " Preview Patient Letter ";
                    btnPrintPrvdrLttr.Text = "Preview Provider Letter";
                    btnSave.Enabled = true;
                    btnInterpretComplete.Enabled = true;
                    HiddenFieldCompletedInterpretation.Value = "0";
                    break;
                case "INTERPRETED":
                    // TODO: Role Needed?
                    // btnSave.Enabled = _userAccountManager.GetRoles().Contains((int)RoleCode.EFR_TEFSC_COORDINATOR);
                    var user = HttpContext.Current.User;

                    btnSave.Enabled = user.IsInRole("CRSADMIN");

                    btnInterpretComplete.Enabled = false;

                    //setup onclick attribute for print buttons when printing is done
                    btnPrintPatntLttr.Attributes.Remove("OnClientClick");
                    btnPrintPrvdrLttr.Attributes.Remove("OnClientClick");

                    HiddenFieldCompletedInterpretation.Value = "1";

                    break;
                default:
                    break;
            }
        }

        protected void BtnSave_Click(object sender, EventArgs e)
        {
            if ((this.OnSaveClick != null))
                this.OnSaveClick(this, new EventArgs());
        }

        protected void BtnInterpretComplete_Click(object sender, EventArgs e)
        {
            if ((this.OnInterpretComplete != null))
                this.OnInterpretComplete(this, new EventArgs());
        }

        protected void BtnPrintPrvdrLttr_Click(object sender, EventArgs e)
        {
            if ((this.OnPrintProviderLetter != null))
                this.OnPrintProviderLetter(this, new EventArgs());
        }

        protected void BtnAddPrintQueue_Click(object sender, EventArgs e)
        {
            if ((this.OnAddPrintQueue != null))
                this.OnAddPrintQueue(this, new EventArgs());
        }

        protected void BtnPrintPatntLttr_Click(object sender, EventArgs e)
        {
            if ((this.OnPrintPatientLetter != null))
                this.OnPrintPatientLetter(this, new EventArgs());
        }

        protected void BtnCancel_Click(object sender, EventArgs e)
        {
            if ((this.OnCancelClick != null))
                this.OnCancelClick(this, new EventArgs());
        }

        public void UpdateInterpretationDetails(string CURRENT_USER, int CURRENT_REGISTRY_ID, WKF_CASE wCase)
        {
            Address ProviderAddress = ProvAddress.GetAddress();
            Address PatntAddress = PatientAddress.GetAddress();

            var activity = WKF_CASE_ACTIVITYManager.GetitemsByWorkflowCaseId(CURRENT_USER, CURRENT_REGISTRY_ID, wCase.WKF_CASE_ID)
                .Where(e => e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation) && e.ARCHIVED_FLAG == false)
                .FirstOrDefault();

            var letter = INTERPRETATION_LETTERManager.GetItemByWkfCaseActivityId(CURRENT_USER, CURRENT_REGISTRY_ID, activity.WKF_CASE_ACTIVITY_ID);
            if (letter == null)
                IntLetter = new INTERPRETATION_LETTER();
            else
                IntLetter = letter;

            IntLetter.WKF_CASE_ACTIVITY_ID = activity.WKF_CASE_ACTIVITY_ID;
            IntLetter.PROVIDER_NAME = ProviderAddress.Name;
            IntLetter.PROVIDER_ADDR1 = ProviderAddress.AddressPart1;
            IntLetter.PROVIDER_ADDR2 = ProviderAddress.AddressPart2;
            IntLetter.PROVIDER_ADDR3 = ProviderAddress.AddressPart3;
            IntLetter.PROVIDER_CITY = ProviderAddress.City;
            IntLetter.PROVIDER_STATE = ProviderAddress.State;
            IntLetter.PROVIDER_COUNTRY = ProviderAddress.Country;
            IntLetter.PROVIDER_PHONE = ProviderAddress.Phone;
            IntLetter.PROVIDER_POSTAL = ProviderAddress.Zip;
            IntLetter.PROVIDER_INSTITUTION_NAME = ProviderAddress.Facility;

            IntLetter.PATIENT_NAME = PatntAddress.Name;
            IntLetter.PATIENT_ADDR1 = PatntAddress.AddressPart1;
            IntLetter.PATIENT_ADDR2 = PatntAddress.AddressPart2;
            IntLetter.PATIENT_ADDR3 = PatntAddress.AddressPart3;
            IntLetter.PATIENT_CITY = PatntAddress.City;
            IntLetter.PATIENT_STATE = PatntAddress.State;
            IntLetter.PATIENT_COUNTRY = PatntAddress.Country;
            IntLetter.PATIENT_PHONE = PatntAddress.Phone;
            IntLetter.PATIENT_POSTAL = PatntAddress.Zip;

            IntLetter.PATIENT_INTERPRETATION_TEXT = TextBoxInterpretationNotesToPatient.Text;
            IntLetter.PROVIDER_INTERPRETATION_TEXT = TextBoxInterpretationNotesToProvider.Text;

            IntLetter.FROM_ADDRESS = TextBoxFrom.Text;
            IntLetter.Letter_SUBJECT = TextBoxSubject.Text;
            IntLetter.STD_SIGNING_CLINICIAN_ID = StringUtility.TryGetValueFromString<int>(DropDownListSigningClinician.SelectedValue);
            IntLetter.CCAddress = TextBoxCCName.Text;

            IntLetter.PROVIDER_COMMENTS = TextBoxSigningCommentsProviderLetter.Text;
            IntLetter.PATIENT_COMMENT = TextBoxSigningCommentsPatientLetter.Text;

            INTERPRETATION_LETTERManager.Save(CURRENT_USER, CURRENT_REGISTRY_ID, IntLetter);
        }

        public void SaveBioFollowUp(string CURRENT_USER, int CURRENT_REGISTRY_ID, WKF_CASE wCase)
        {
            WKF_CASE_ACTIVITY activity = WKF_CASE_ACTIVITYManager.GetitemsByWorkflowCaseId(CURRENT_USER, CURRENT_REGISTRY_ID, wCase.WKF_CASE_ID)
                .Where(e => e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation) && e.ARCHIVED_FLAG == false)
                .FirstOrDefault();

            if (activity != null)
            {
                activity.FOLLOWUP_REQUIRED_FLAG = (int.Parse(rbNeedBiomonitoringFollowUp.SelectedValue) == 1);

                activity.FOLLOWUP_PLAN_DATE = StringUtility.TryGetValueFromString<DateTime>(txtBioMonDueDate.Text.Trim());

                activity.FOLLOWUP_REMINDER_DATE = StringUtility.TryGetValueFromString<DateTime>(txtReminderDate.Text.Trim());

                activity.WKF_CASE_ACTIVITY_ID = WKF_CASE_ACTIVITYManager.Save(CURRENT_USER, CURRENT_REGISTRY_ID, activity);
            }
        }

        public void UpdateProviderAddress(string CURRENT_USER, int CURRENT_REGISTRY_ID, VIRP.CRS.BO.REFERRAL referral)
        {
            Address ProviderAddress = ProvAddress.GetAddress();

            var rerferralReg = REFERRAL_REGManager.GetItemByRegistryPatient(CURRENT_USER, CURRENT_REGISTRY_ID, referral.PATIENT_ID);

            var wCase = VIRP.CRS.BLL.WKF_CASEManager.GetItemsByReferral(CURRENT_USER, CURRENT_REGISTRY_ID, referral.REFERRAL_ID).FirstOrDefault();

            var activity = WKF_CASE_ACTIVITYManager.GetitemsByWorkflowCaseId(CURRENT_USER, CURRENT_REGISTRY_ID, wCase.WKF_CASE_ID)
                .Where(e => e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation) && e.ARCHIVED_FLAG == false)
                .FirstOrDefault();

            var letter = INTERPRETATION_LETTERManager.GetItemByWkfCaseActivityId(CURRENT_USER, CURRENT_REGISTRY_ID, activity.WKF_CASE_ACTIVITY_ID);

            if (activity != null && letter != null)
            {
                // Got it from interpretation letters table
                IntLetter = letter;
                UseItToCompareAddress = GetLetterProviderAddress(IntLetter);
            }
            else  // Got it from Referral table
                UseItToCompareAddress = GetPrimaryCareProviderAddress(rerferralReg);
            
            // Check if the pre-populated provider address has been changed.
            // TODO - If Address line 1 exists in activity, use it to compare. Else use referral
            if (UseItToCompareAddress.AddressPart1 != ProviderAddress.AddressPart1 || UseItToCompareAddress.AddressPart2 != ProviderAddress.AddressPart2 ||
                UseItToCompareAddress.AddressPart3 != ProviderAddress.AddressPart3 || UseItToCompareAddress.City != ProviderAddress.City ||
                UseItToCompareAddress.Country != ProviderAddress.Country || UseItToCompareAddress.State != ProviderAddress.State ||
                UseItToCompareAddress.Name != ProviderAddress.Name || UseItToCompareAddress.Phone != ProviderAddress.Phone || UseItToCompareAddress.Zip != ProviderAddress.Zip)
            {
                // Insert a new record in INTERPRETATION_LETTER
                if (letter != null)
                {
                    IntLetter = new INTERPRETATION_LETTER();
                    INTERPRETATION_LETTERManager.Save(CURRENT_USER, CURRENT_REGISTRY_ID, IntLetter);
                    letter = INTERPRETATION_LETTERManager.GetItemByWkfCaseActivityId(CURRENT_USER, CURRENT_REGISTRY_ID, activity.WKF_CASE_ACTIVITY_ID);
                }

                IntLetter = letter;

                IntLetter.WKF_CASE_ACTIVITY_ID = activity.WKF_CASE_ACTIVITY_ID;
                IntLetter.PROVIDER_NAME = ProviderAddress.Name;
                IntLetter.PROVIDER_ADDR1 = ProviderAddress.AddressPart1;
                IntLetter.PROVIDER_ADDR2 = ProviderAddress.AddressPart2;
                IntLetter.PROVIDER_ADDR3 = ProviderAddress.AddressPart3;
                IntLetter.PROVIDER_CITY = ProviderAddress.City;
                IntLetter.PROVIDER_STATE = ProviderAddress.State;
                IntLetter.PROVIDER_COUNTRY = ProviderAddress.Country;
                IntLetter.PROVIDER_PHONE = ProviderAddress.Phone;
                IntLetter.PROVIDER_POSTAL = ProviderAddress.Zip;
            }
        }
    }
}